/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  HttpStatusCode,
  TalerMerchantApi,
  isRfc3548Base32Charset,
  randomRfc3548Base32Key,
} from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../../../components/form/FormProvider.js";
import { Input } from "../../../../components/form/Input.js";
import { InputSelector } from "../../../../components/form/InputSelector.js";
import { InputWithAddon } from "../../../../components/form/InputWithAddon.js";
import { undefinedIfEmpty } from "../../../../utils/table.js";
import { useSessionContext } from "../../../../context/session.js";

const TALER_SCREEN_ID = 50;

type Entity = TalerMerchantApi.OtpDeviceAddDetails;

interface Props {
  onCreated: (d: TalerMerchantApi.OtpDeviceAddDetails) => void;
  onBack?: () => void;
}

const algorithms = [0, 1];
const algorithmsNames = ["30s 8d TOTP-SHA1", "30s 8d eTOTP-SHA1"];

export function CreatePage({ onCreated, onBack }: Props): VNode {
  const { i18n } = useTranslationContext();

  const [state, setState] = useState<Partial<Entity>>({
    otp_key: "E5V6OSMTZRL2ARYKNDJ4SUDGYOCLQUOT",
    otp_ctr: 1,
    otp_device_id: "1",
    otp_device_description: "asd",
    otp_algorithm: 0,
  });
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const { state: session, lib } = useSessionContext();

  const [showKey, setShowKey] = useState(false);

  const errors = undefinedIfEmpty<FormErrors<Entity>>({
    otp_device_id: !state.otp_device_id
      ? i18n.str`Required`
      : !/[a-zA-Z0-9]*/.test(state.otp_device_id)
        ? i18n.str`Invalid. Please enter letters and numbers only.`
        : undefined,
    // otp_algorithm: !state.otp_algorithm ? i18n.str`Required` : undefined,
    otp_key: !state.otp_key
      ? i18n.str`Required`
      : !isRfc3548Base32Charset(state.otp_key)
        ? i18n.str`Just letters and numbers from 2 to 7`
        : state.otp_key.length !== 32
          ? i18n.str`The size of the key must be 32 characters`
          : undefined,
    otp_device_description: !state.otp_device_description
      ? i18n.str`Required`
      : !/[a-zA-Z0-9]*/.test(state.otp_device_description)
        ? i18n.str`Invalid. Please enter letters and numbers only.`
        : undefined,
  });

  const hasErrors = errors !== undefined;

  const data = hasErrors
    ? undefined
    : (state as TalerMerchantApi.OtpDeviceAddDetails);
  const create = safeFunctionHandler(
    lib.instance.addOtpDevice.bind(lib.instance),
    !session.token || !data ? undefined : [session.token, data],
  );
  create.onSuccess = (success, token, req) => {
    onCreated(req);
  };
  create.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section is-main-section">
        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <FormProvider
              object={state}
              valueHandler={setState}
              errors={errors}
            >
              <Input<Entity>
                name="otp_device_id"
                label={i18n.str`ID`}
                tooltip={i18n.str`Internal ID on the system`}
              />
              <Input<Entity>
                name="otp_device_description"
                label={i18n.str`Description`}
                tooltip={i18n.str`Useful to identify the device physically`}
              />
              <InputSelector<Entity>
                name="otp_algorithm"
                label={i18n.str`Verification algorithm`}
                tooltip={i18n.str`Algorithm used to verify transactions in offline mode`}
                values={algorithms}
                toStr={(v) => algorithmsNames[v]}
                fromStr={(v) => Number(v)}
              />
              <InputWithAddon<Entity>
                expand
                name="otp_key"
                label={i18n.str`Device key`}
                inputType={showKey ? "text" : "password"}
                help={i18n.str`Be sure to choose a password that is hard to guess, or use the random generator.`}
                tooltip={i18n.str`Your device needs to match exactly the same value`}
                fromStr={(v) => v.toUpperCase()}
                addonAfterAction={() => {
                  setShowKey(!showKey);
                }}
                addonAfter={
                  <span class="icon">
                    {showKey ? (
                      <i class="mdi mdi-eye" />
                    ) : (
                      <i class="mdi mdi-eye-off" />
                    )}
                  </span>
                }
                side={
                  <button
                    type="button"
                    data-tooltip={i18n.str`Generate random secret key`}
                    class="button is-info mr-3"
                    onClick={(e) => {
                      setState((s) => ({
                        ...s,
                        otp_key: randomRfc3548Base32Key(),
                      }));
                      e.preventDefault();
                    }}
                  >
                    <i18n.Translate>Random</i18n.Translate>
                  </button>
                }
              />

              <div class="buttons is-right mt-5">
                {onBack && (
                  <button class="button" type="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </button>
                )}
                <ButtonBetterBulma
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields`
                      : i18n.str`Confirm operation`
                  }
                  onClick={create}
                  type="submit"
                >
                  <i18n.Translate>Confirm</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </FormProvider>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
