/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.loader;

import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutorService;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hugegraph.driver.HugeClient;
import org.apache.hugegraph.exception.ServerException;
import org.apache.hugegraph.loader.constant.Constants;
import org.apache.hugegraph.loader.constant.ElemType;
import org.apache.hugegraph.loader.exception.InitException;
import org.apache.hugegraph.loader.exception.LoadException;
import org.apache.hugegraph.loader.exception.ParseException;
import org.apache.hugegraph.loader.exception.ReadException;
import org.apache.hugegraph.loader.executor.GroovyExecutor;
import org.apache.hugegraph.loader.executor.LoadContext;
import org.apache.hugegraph.loader.executor.LoadOptions;
import org.apache.hugegraph.loader.filter.util.SchemaManagerProxy;
import org.apache.hugegraph.loader.filter.util.ShortIdConfig;
import org.apache.hugegraph.loader.mapping.ElementMapping;
import org.apache.hugegraph.loader.mapping.InputStruct;
import org.apache.hugegraph.loader.mapping.LoadMapping;
import org.apache.hugegraph.loader.metrics.LoadMetrics;
import org.apache.hugegraph.loader.metrics.LoadSummary;
import org.apache.hugegraph.loader.progress.InputProgress;
import org.apache.hugegraph.loader.reader.InputReader;
import org.apache.hugegraph.loader.reader.line.Line;
import org.apache.hugegraph.loader.source.SourceType;
import org.apache.hugegraph.loader.source.graph.GraphSource;
import org.apache.hugegraph.loader.task.GlobalExecutorManager;
import org.apache.hugegraph.loader.task.ParseTaskBuilder;
import org.apache.hugegraph.loader.task.TaskManager;
import org.apache.hugegraph.loader.util.HugeClientHolder;
import org.apache.hugegraph.loader.util.LoadUtil;
import org.apache.hugegraph.loader.util.Printer;
import org.apache.hugegraph.structure.constant.HugeType;
import org.apache.hugegraph.structure.schema.EdgeLabel;
import org.apache.hugegraph.structure.schema.IndexLabel;
import org.apache.hugegraph.structure.schema.PropertyKey;
import org.apache.hugegraph.structure.schema.SchemaLabel;
import org.apache.hugegraph.structure.schema.VertexLabel;
import org.apache.hugegraph.util.ExecutorUtil;
import org.apache.hugegraph.util.JsonUtil;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public final class HugeGraphLoader {
    public static final Logger LOG = Log.logger(HugeGraphLoader.class);
    private final LoadContext context;
    private final LoadMapping mapping;
    private final TaskManager manager;
    private final LoadOptions options;

    public static void main(String[] args) {
        HugeGraphLoader loader;
        try {
            loader = new HugeGraphLoader(args);
        }
        catch (Throwable e) {
            Printer.printError("Failed to start loading", e);
            System.exit(1);
            return;
        }
        try {
            loader.load();
        }
        finally {
            loader.shutdown();
            GlobalExecutorManager.shutdown(loader.options.shutdownTimeout);
        }
    }

    public HugeGraphLoader(String[] args) {
        this(LoadOptions.parseOptions(args));
    }

    public HugeGraphLoader(LoadOptions options) {
        this(options, LoadMapping.of(options.file));
        GlobalExecutorManager.setBatchThreadCount(options.batchInsertThreads);
        GlobalExecutorManager.setSingleThreadCount(options.singleInsertThreads);
    }

    public HugeGraphLoader(LoadOptions options, LoadMapping mapping) {
        this.context = new LoadContext(options);
        this.options = options;
        this.mapping = mapping;
        this.manager = new TaskManager(this.context);
        this.addShutdownHook();
    }

    private void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            LOG.info("Shutdown hook was triggered");
            this.stopThenShutdown();
        }));
    }

    public LoadContext context() {
        return this.context;
    }

    private void checkGraphExists() {
        HugeClient client = this.context.indirectClient();
        String targetGraph = this.options.graph;
        if (this.options.createGraph && !client.graphs().listGraph().contains(targetGraph)) {
            HashMap<String, String> conf = new HashMap<String, String>();
            conf.put("store", targetGraph);
            conf.put("backend", this.options.backend);
            conf.put("serializer", this.options.serializer);
            conf.put("task.scheduler_type", this.options.schedulerType);
            conf.put("nickname", targetGraph);
            client.graphs().createGraph(targetGraph, JsonUtil.toJson(conf));
            LOG.info("Create graph " + targetGraph + " ......");
        }
    }

    private void setGraphMode() {
        Supplier<Stream> inputsSupplier = () -> this.mapping.structs().stream().filter(struct -> !struct.skip()).map(InputStruct::input);
        boolean allMatch = inputsSupplier.get().allMatch(input -> SourceType.GRAPH.equals((Object)input.type()));
        boolean anyMatch = inputsSupplier.get().anyMatch(input -> SourceType.GRAPH.equals((Object)input.type()));
        if (anyMatch && !allMatch) {
            throw new LoadException("All inputs must be of Graph Type");
        }
        if (allMatch || this.options.restore) {
            this.context().setRestoreMode();
        } else {
            this.context().setLoadingMode();
        }
    }

    public boolean load() {
        this.options.dumpParams();
        try {
            this.checkGraphExists();
            this.setGraphMode();
            this.clearAllDataIfNeeded();
            this.createSchema();
            this.loadInputs();
            Printer.printSummary(this.context);
        }
        catch (Throwable t) {
            this.context.occurredError();
            if (t instanceof ServerException) {
                ServerException e = (ServerException)t;
                String logMessage = "Log ServerException: \n" + e.exception() + "\n";
                if (e.trace() != null) {
                    logMessage = logMessage + StringUtils.join((Iterable)((List)e.trace()), (String)"\n");
                }
                LOG.warn(logMessage);
            }
            throw LoadUtil.targetRuntimeException(t);
        }
        return true;
    }

    public void shutdown() {
        this.stopThenShutdown();
    }

    private void clearAllDataIfNeeded() {
        if (!this.options.clearAllData) {
            return;
        }
        int requestTimeout = this.options.timeout;
        this.options.timeout = this.options.clearTimeout;
        HugeClient client = HugeClientHolder.create(this.options);
        try {
            LOG.info("Prepare to clear the data of graph '{}'", (Object)this.options.graph);
            client.graphs().clearGraph(this.options.graph, "I'm sure to delete all data");
            LOG.info("The graph '{}' has been cleared successfully", (Object)this.options.graph);
        }
        catch (Exception e) {
            LOG.error("Failed to clear data for graph '{}': {}", new Object[]{this.options.graph, e.getMessage(), e});
            throw e;
        }
        finally {
            this.options.timeout = requestTimeout;
        }
    }

    private void createSchema() {
        if (!StringUtils.isEmpty((CharSequence)this.options.schema)) {
            String script;
            File file = FileUtils.getFile((String[])new String[]{this.options.schema});
            HugeClient client = this.context.client();
            GroovyExecutor groovyExecutor = new GroovyExecutor();
            if (!this.options.shorterIDConfigs.isEmpty()) {
                SchemaManagerProxy.proxy(client, this.options);
            }
            groovyExecutor.bind("schema", client.schema());
            try {
                script = FileUtils.readFileToString((File)file, (Charset)Constants.CHARSET);
            }
            catch (IOException e) {
                throw new LoadException("Failed to read schema file '%s'", (Throwable)e, this.options.schema);
            }
            if (!this.options.shorterIDConfigs.isEmpty()) {
                for (ShortIdConfig config : this.options.shorterIDConfigs) {
                    PropertyKey propertyKey = (PropertyKey)client.schema().propertyKey(config.getIdFieldName()).ifNotExist().dataType(config.getIdFieldType()).build();
                    client.schema().addPropertyKey(propertyKey);
                }
                groovyExecutor.execute(script, client);
                List vertexLabels = client.schema().getVertexLabels();
                for (VertexLabel vertexLabel : vertexLabels) {
                    ShortIdConfig config = this.options.getShortIdConfig(vertexLabel.name());
                    if (config == null) continue;
                    config.setLabelID(vertexLabel.id());
                    IndexLabel indexLabel = (IndexLabel)client.schema().indexLabel(config.getVertexLabel() + "By" + config.getIdFieldName()).onV(config.getVertexLabel()).by(new String[]{config.getIdFieldName()}).secondary().ifNotExist().build();
                    client.schema().addIndexLabel(indexLabel);
                }
            } else {
                groovyExecutor.execute(script, client);
            }
        }
        List<InputStruct> structs = this.mapping.structs();
        for (InputStruct struct : structs) {
            if (!SourceType.GRAPH.equals((Object)struct.input().type())) continue;
            GraphSource graphSouce = (GraphSource)struct.input();
            if (StringUtils.isEmpty((CharSequence)graphSouce.getPdPeers())) {
                graphSouce.setPdPeers(this.options.pdPeers);
            }
            if (StringUtils.isEmpty((CharSequence)graphSouce.getMetaEndPoints())) {
                graphSouce.setMetaEndPoints(this.options.metaEndPoints);
            }
            if (StringUtils.isEmpty((CharSequence)graphSouce.getCluster())) {
                graphSouce.setCluster(this.options.cluster);
            }
            if (StringUtils.isEmpty((CharSequence)graphSouce.getUsername())) {
                graphSouce.setUsername(this.options.username);
            }
            if (StringUtils.isEmpty((CharSequence)graphSouce.getPassword())) {
                graphSouce.setPassword(this.options.password);
            }
            GraphSource graphSource = (GraphSource)struct.input();
            this.createGraphSourceSchema(graphSource);
        }
        this.context.updateSchemaCache();
    }

    private void createGraphSourceSchema(GraphSource graphSource) {
        try (HugeClient sourceClient = graphSource.createHugeClient();
             HugeClient client = HugeClientHolder.create(this.options, false);){
            this.createGraphSourceVertexLabel(sourceClient, client, graphSource);
            this.createGraphSourceEdgeLabel(sourceClient, client, graphSource);
            this.createGraphSourceIndexLabel(sourceClient, client, graphSource);
        }
        catch (Exception e) {
            LOG.error("Failed to create graph source schema for {}: {}", new Object[]{graphSource.getGraph(), e.getMessage(), e});
            throw new LoadException("Schema creation failed", e);
        }
    }

    private void createGraphSourceLabels(HugeClient sourceClient, HugeClient targetClient, List<? extends SchemaLabel> labels, Map<String, GraphSource.SelectedLabelDes> selectedMap, Map<String, GraphSource.IgnoredLabelDes> ignoredMap, boolean isVertex) {
        for (SchemaLabel schemaLabel : labels) {
            GraphSource.IgnoredLabelDes des;
            if (ignoredMap.containsKey(schemaLabel.name()) && (des = ignoredMap.get(schemaLabel.name())).getProperties() != null) {
                des.getProperties().forEach(p -> label.properties().remove(p));
            }
            Set existedPKs = targetClient.schema().getPropertyKeys().stream().map(pk -> pk.name()).collect(Collectors.toSet());
            for (String pkName : schemaLabel.properties()) {
                PropertyKey pk2 = sourceClient.schema().getPropertyKey(pkName);
                if (existedPKs.contains(pk2.name())) continue;
                targetClient.schema().addPropertyKey(pk2);
            }
            if (isVertex) {
                if (!(schemaLabel instanceof VertexLabel)) {
                    throw new IllegalArgumentException("Expected VertexLabel but got " + schemaLabel.getClass());
                }
                targetClient.schema().addVertexLabel((VertexLabel)schemaLabel);
                continue;
            }
            if (!(schemaLabel instanceof EdgeLabel)) {
                throw new IllegalArgumentException("Expected EdgeLabel but got " + schemaLabel.getClass());
            }
            targetClient.schema().addEdgeLabel((EdgeLabel)schemaLabel);
        }
    }

    private void createGraphSourceVertexLabel(HugeClient sourceClient, HugeClient targetClient, GraphSource graphSource) {
        sourceClient.assignGraph(graphSource.getGraphSpace(), graphSource.getGraph());
        List vertexLabels = new ArrayList();
        if (graphSource.getSelectedVertices() != null) {
            List selectedVertexLabels = graphSource.getSelectedVertices().stream().map(des -> des.getLabel()).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(selectedVertexLabels)) {
                vertexLabels = sourceClient.schema().getVertexLabels(selectedVertexLabels);
            }
        } else {
            vertexLabels = sourceClient.schema().getVertexLabels();
        }
        HashMap<String, GraphSource.SelectedLabelDes> mapSelectedVertices = new HashMap<String, GraphSource.SelectedLabelDes>();
        if (graphSource.getSelectedVertices() != null) {
            for (GraphSource.SelectedLabelDes des2 : graphSource.getSelectedVertices()) {
                mapSelectedVertices.put(des2.getLabel(), des2);
            }
        }
        for (VertexLabel label : vertexLabels) {
            List<String> selectedProperties;
            if (mapSelectedVertices.getOrDefault(label.name(), null) == null || (selectedProperties = ((GraphSource.SelectedLabelDes)mapSelectedVertices.get(label.name())).getProperties()) == null) continue;
            label.properties().clear();
            label.properties().addAll(selectedProperties);
        }
        HashMap<String, GraphSource.IgnoredLabelDes> mapIgnoredVertices = new HashMap<String, GraphSource.IgnoredLabelDes>();
        if (graphSource.getIgnoredVertices() != null) {
            for (GraphSource.IgnoredLabelDes des3 : graphSource.getIgnoredVertices()) {
                mapIgnoredVertices.put(des3.getLabel(), des3);
            }
        }
        this.createGraphSourceLabels(sourceClient, targetClient, vertexLabels, mapSelectedVertices, mapIgnoredVertices, true);
    }

    private void createGraphSourceEdgeLabel(HugeClient sourceClient, HugeClient targetClient, GraphSource graphSource) {
        List edgeLabels = new ArrayList();
        if (graphSource.getSelectedEdges() != null) {
            List selectedEdgeLabels = graphSource.getSelectedEdges().stream().map(des -> des.getLabel()).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(selectedEdgeLabels)) {
                edgeLabels = sourceClient.schema().getEdgeLabels(selectedEdgeLabels);
            }
        } else {
            edgeLabels = sourceClient.schema().getEdgeLabels();
        }
        HashMap<String, GraphSource.SelectedLabelDes> mapSelectedEdges = new HashMap<String, GraphSource.SelectedLabelDes>();
        if (graphSource.getSelectedEdges() != null) {
            for (GraphSource.SelectedLabelDes des2 : graphSource.getSelectedEdges()) {
                mapSelectedEdges.put(des2.getLabel(), des2);
            }
        }
        for (EdgeLabel label : edgeLabels) {
            List<String> selectedProperties;
            if (mapSelectedEdges.getOrDefault(label.name(), null) == null || (selectedProperties = ((GraphSource.SelectedLabelDes)mapSelectedEdges.get(label.name())).getProperties()) == null) continue;
            label.properties().clear();
            label.properties().addAll(selectedProperties);
        }
        HashMap<String, GraphSource.IgnoredLabelDes> mapIgnoredEdges = new HashMap<String, GraphSource.IgnoredLabelDes>();
        if (graphSource.getIgnoredEdges() != null) {
            for (GraphSource.IgnoredLabelDes des3 : graphSource.getIgnoredEdges()) {
                mapIgnoredEdges.put(des3.getLabel(), des3);
            }
        }
        this.createGraphSourceLabels(sourceClient, targetClient, edgeLabels, mapSelectedEdges, mapIgnoredEdges, false);
    }

    private void createGraphSourceIndexLabel(HugeClient sourceClient, HugeClient targetClient, GraphSource graphSource) {
        Set existedVertexLabels = targetClient.schema().getVertexLabels().stream().map(v -> v.name()).collect(Collectors.toSet());
        Set existedEdgeLabels = targetClient.schema().getEdgeLabels().stream().map(v -> v.name()).collect(Collectors.toSet());
        List indexLabels = sourceClient.schema().getIndexLabels();
        for (IndexLabel indexLabel : indexLabels) {
            Set curFields;
            HugeType baseType = indexLabel.baseType();
            String baseValue = indexLabel.baseValue();
            HashSet sourceIndexFields = new HashSet(indexLabel.indexFields());
            if (baseType.equals((Object)HugeType.VERTEX_LABEL) && existedVertexLabels.contains(baseValue) && (curFields = targetClient.schema().getVertexLabel(baseValue).properties()).containsAll(sourceIndexFields)) {
                targetClient.schema().addIndexLabel(indexLabel);
            }
            if (!baseType.equals((Object)HugeType.EDGE_LABEL) || !existedEdgeLabels.contains(baseValue) || !(curFields = targetClient.schema().getEdgeLabel(baseValue).properties()).containsAll(sourceIndexFields)) continue;
            targetClient.schema().addIndexLabel(indexLabel);
        }
    }

    private void loadInputs() {
        Printer.printRealtimeProgress(this.context);
        LoadOptions options = this.context.options();
        LoadSummary summary = this.context.summary();
        summary.initMetrics(this.mapping);
        summary.startTotalTimer();
        try {
            if (!options.failureMode) {
                this.loadInputs(this.mapping.structs());
            } else {
                this.loadInputs(this.mapping.structsForFailure(options));
            }
            this.manager.waitFinished();
        }
        finally {
            summary.calculateTotalTime(ElemType.VERTEX);
            summary.calculateTotalTime(ElemType.EDGE);
            summary.stopTotalTimer();
        }
        Printer.printFinalProgress(this.context);
    }

    private void loadInputs(List<InputStruct> structs) {
        if (this.context.options().checkVertex) {
            LOG.info("Forced to load vertices before edges since set option check-vertex=true");
            SplitInputStructs split = this.splitStructs(structs);
            this.loadStructs(split.vertexInputStructs);
            this.manager.waitFinished("vertex insert tasks");
            this.loadStructs(split.edgeInputStructs);
        } else {
            this.loadStructs(structs);
        }
    }

    private List<InputTaskItem> prepareTaskItems(List<InputStruct> structs, boolean scatter) {
        ArrayList<InputTaskItem> tasks = new ArrayList<InputTaskItem>();
        ArrayList readers = new ArrayList();
        int curFile = 0;
        int curIndex = 0;
        for (InputStruct struct : structs) {
            if (struct.skip()) continue;
            try {
                LOG.info("Start loading: '{}'", (Object)struct);
                InputReader reader = InputReader.create(struct.input());
                ImmutableList readerList = reader.multiReaders() ? reader.split() : ImmutableList.of((Object)reader);
                readers.addAll(readerList);
                LOG.info("total {} found in '{}'", (Object)readerList.size(), (Object)struct);
                tasks.ensureCapacity(tasks.size() + readerList.size());
                int seq = 0;
                for (InputReader r : readerList) {
                    if (curFile >= this.context.options().startFile && (this.context.options().endFile == -1 || curFile < this.context.options().endFile)) {
                        tasks.add(new InputTaskItem(struct, r, seq, curIndex));
                    } else {
                        r.close();
                    }
                    ++seq;
                    ++curFile;
                }
                if (this.context.options().endFile != -1 && curFile >= this.context.options().endFile) {
                    break;
                }
            }
            catch (InitException e) {
                throw new LoadException("Failed to init input reader", e);
            }
            finally {
                Set usedReaders = tasks.stream().map(item -> item.reader).collect(Collectors.toSet());
                for (InputReader r : readers) {
                    if (usedReaders.contains(r)) continue;
                    try {
                        r.close();
                    }
                    catch (Exception ex) {
                        LOG.warn("Failed to close reader", (Throwable)ex);
                    }
                }
            }
            ++curIndex;
        }
        if (scatter) {
            tasks.sort(Comparator.comparingInt(o -> o.structIndex).thenComparingInt(o -> o.seqNumber));
        }
        return tasks;
    }

    private void loadStructs(List<InputStruct> structs) {
        int parallelCount = this.context.options().parallelCount;
        if (structs.size() == 0) {
            return;
        }
        if (parallelCount <= 0) {
            parallelCount = Math.min(structs.size(), Runtime.getRuntime().availableProcessors() * 2);
        }
        boolean scatter = this.context.options().scatterSources;
        LOG.info("{} threads for loading {} structs, from {} to {} in {} mode", new Object[]{parallelCount, structs.size(), this.context.options().startFile, this.context.options().endFile, scatter ? "scatter" : "sequential"});
        ExecutorService loadService = null;
        try {
            loadService = ExecutorUtil.newFixedThreadPool((int)parallelCount, (String)"loader");
            List<InputTaskItem> taskItems = this.prepareTaskItems(structs, scatter);
            ArrayList<CompletableFuture<Void>> loadTasks = new ArrayList<CompletableFuture<Void>>();
            if (taskItems.isEmpty()) {
                LOG.info("No tasks to execute after filtering");
                return;
            }
            for (InputTaskItem item : taskItems) {
                item.reader.init(this.context, item.struct);
                loadTasks.add(this.asyncLoadStruct(item.struct, item.reader, loadService));
            }
            LOG.info("waiting for loading finish {}", (Object)loadTasks.size());
            CompletableFuture.allOf(loadTasks.toArray(new CompletableFuture[0])).join();
        }
        catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof ParseException) {
                throw (ParseException)cause;
            }
            if (cause instanceof LoadException) {
                throw (LoadException)cause;
            }
            if (cause != null) {
                if (cause instanceof RuntimeException) {
                    throw (RuntimeException)cause;
                }
                throw new RuntimeException(cause);
            }
            throw e;
        }
        catch (Throwable t) {
            throw t;
        }
        finally {
            this.cleanupEmptyProgress();
            if (loadService != null) {
                loadService.shutdownNow();
            }
            LOG.info("Load end");
        }
    }

    private CompletableFuture<Void> asyncLoadStruct(InputStruct struct, InputReader reader, ExecutorService service) {
        return CompletableFuture.runAsync(() -> {
            try {
                this.loadStruct(struct, reader);
            }
            catch (Throwable t) {
                throw t;
            }
            finally {
                reader.close();
            }
        }, service);
    }

    private void loadStruct(InputStruct struct, InputReader reader) {
        LOG.info("Start loading '{}'", (Object)struct);
        LoadMetrics metrics = this.context.summary().metrics(struct);
        metrics.startInFlight();
        ParseTaskBuilder taskBuilder = new ParseTaskBuilder(this.context, struct);
        int batchSize = this.context.options().batchSize;
        ArrayList<Line> lines = new ArrayList<Line>(batchSize);
        long batchStartTime = System.currentTimeMillis();
        boolean finished = false;
        while (!finished && !this.context.stopped()) {
            try {
                if (reader.hasNext()) {
                    Line next = (Line)reader.next();
                    if (next != null) {
                        lines.add(next);
                        metrics.increaseReadSuccess();
                    }
                } else {
                    finished = true;
                }
            }
            catch (ReadException e) {
                metrics.increaseReadFailure();
                this.handleReadFailure(struct, e);
            }
            boolean reachedMaxReadLines = this.reachedMaxReadLines();
            if (reachedMaxReadLines) {
                finished = true;
            }
            if (lines.size() < batchSize && (lines.size() <= 0 || System.currentTimeMillis() <= batchStartTime + 5000L) && !finished) continue;
            List<ParseTaskBuilder.ParseTask> tasks = taskBuilder.build(lines);
            for (ParseTaskBuilder.ParseTask task : tasks) {
                this.executeParseTask(struct, task.mapping(), task);
            }
            reader.confirmOffset();
            this.context.newProgress().markLoaded(struct, reader, finished);
            this.handleParseFailure();
            if (reachedMaxReadLines) {
                LOG.warn("Read lines exceed limit, stopped loading tasks");
                this.context.stopLoading();
            }
            lines = new ArrayList(batchSize);
            batchStartTime = System.currentTimeMillis();
        }
        metrics.stopInFlight();
        LOG.info("Finish loading '{}'", (Object)struct);
    }

    private void executeParseTask(InputStruct struct, ElementMapping mapping, ParseTaskBuilder.ParseTask task) {
        long start = System.currentTimeMillis();
        Object batches = task.get();
        long end = System.currentTimeMillis();
        this.context.summary().addTimeRange(mapping.type(), start, end);
        if (this.context.options().dryRun || CollectionUtils.isEmpty((Collection)batches)) {
            return;
        }
        Iterator iterator = batches.iterator();
        while (iterator.hasNext()) {
            List batch = (List)iterator.next();
            this.manager.submitBatch(struct, mapping, batch);
        }
    }

    private void handleReadFailure(InputStruct struct, ReadException e) {
        LOG.error("Read {} error", (Object)struct, (Object)e);
        this.context.occurredError();
        LoadOptions options = this.context.options();
        if (options.testMode) {
            throw e;
        }
        this.context.failureLogger(struct).write(e);
        long failures = this.context.summary().totalReadFailures();
        if (options.maxReadErrors != -1 && failures >= (long)options.maxReadErrors) {
            Printer.printError("More than %s read error, stop reading and waiting all parse/insert tasks stopped", options.maxReadErrors);
            this.context.stopLoading();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleParseFailure() {
        LoadOptions options = this.context.options();
        long failures = this.context.summary().totalParseFailures();
        if (options.maxParseErrors != -1 && failures >= (long)options.maxParseErrors) {
            if (this.context.stopped()) {
                return;
            }
            LoadContext loadContext = this.context;
            synchronized (loadContext) {
                if (!this.context.stopped()) {
                    Printer.printError("More than %s parse error, stop parsing and waiting all insert tasks stopped", options.maxParseErrors);
                    this.context.stopLoading();
                }
            }
        }
    }

    private SplitInputStructs splitStructs(List<InputStruct> structs) {
        InputStruct result;
        SplitInputStructs split = new SplitInputStructs();
        for (InputStruct struct : structs) {
            result = struct.extractVertexStruct();
            if (result == InputStruct.EMPTY) continue;
            split.vertexInputStructs.add(result);
        }
        for (InputStruct struct : structs) {
            result = struct.extractEdgeStruct();
            if (result == InputStruct.EMPTY) continue;
            split.edgeInputStructs.add(result);
        }
        return split;
    }

    private boolean reachedMaxReadLines() {
        long maxReadLines = this.context.options().maxReadLines;
        if (maxReadLines == -1L) {
            return false;
        }
        return this.context.summary().totalReadLines() >= maxReadLines;
    }

    private synchronized void stopThenShutdown() {
        if (this.context.closed()) {
            return;
        }
        LOG.info("Stop loading then shutdown HugeGraphLoader");
        try {
            this.context.stopLoading();
            if (this.manager != null) {
                this.manager.waitFinished();
                this.manager.shutdown();
            }
        }
        finally {
            try {
                this.context.unsetLoadingMode();
            }
            finally {
                this.context.close();
            }
        }
    }

    private void cleanupEmptyProgress() {
        Map<String, InputProgress> inputProgressMap = this.context.newProgress().inputProgress();
        inputProgressMap.entrySet().removeIf(entry -> ((InputProgress)entry.getValue()).loadedItems().isEmpty());
    }

    private static class SplitInputStructs {
        private final List<InputStruct> vertexInputStructs = new ArrayList<InputStruct>();
        private final List<InputStruct> edgeInputStructs = new ArrayList<InputStruct>();
    }

    public static class InputTaskItem {
        public final InputReader reader;
        public final InputStruct struct;
        public final int structIndex;
        public final int seqNumber;

        public InputTaskItem(InputStruct struct, InputReader reader, int structIndex, int seq) {
            this.struct = struct;
            this.reader = reader;
            this.structIndex = structIndex;
            this.seqNumber = seq;
        }
    }
}

