/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.gravitino.listener.api.event.job;

import org.apache.gravitino.annotation.DeveloperApi;
import org.apache.gravitino.listener.api.event.OperationType;
import org.apache.gravitino.utils.NameIdentifierUtil;

/** Represents an event triggered when the retrieval of a job template has failed. */
@DeveloperApi
public class GetJobTemplateFailureEvent extends JobTemplateFailureEvent {

  /**
   * Constructs a new {@code GetJobTemplateFailureEvent} instance.
   *
   * @param user The user who initiated the job template retrieval operation.
   * @param metalake The metalake name where the job template resides.
   * @param jobTemplateName The name of the job template that failed to be retrieved.
   * @param exception The exception encountered during the job template retrieval, providing
   *     insights into the reasons behind the failure.
   */
  public GetJobTemplateFailureEvent(
      String user, String metalake, String jobTemplateName, Exception exception) {
    super(user, NameIdentifierUtil.ofJobTemplate(metalake, jobTemplateName), exception);
  }

  /**
   * Returns the type of operation.
   *
   * @return the operation type.
   */
  @Override
  public OperationType operationType() {
    return OperationType.GET_JOB_TEMPLATE;
  }
}
