/*
 * Decompiled with CFR 0.152.
 */
package org.apache.dolphinscheduler.plugin.storage.gcs;

import com.google.api.gax.paging.Page;
import com.google.auth.Credentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Bucket;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.Generated;
import org.apache.commons.lang3.StringUtils;
import org.apache.dolphinscheduler.common.utils.FileUtils;
import org.apache.dolphinscheduler.plugin.storage.api.AbstractStorageOperator;
import org.apache.dolphinscheduler.plugin.storage.api.ResourceMetadata;
import org.apache.dolphinscheduler.plugin.storage.api.StorageEntity;
import org.apache.dolphinscheduler.plugin.storage.api.StorageOperator;
import org.apache.dolphinscheduler.plugin.storage.gcs.GcsStorageProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GcsStorageOperator
extends AbstractStorageOperator
implements Closeable,
StorageOperator {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(GcsStorageOperator.class);
    private final Storage gcsStorage;
    private final String bucketName;

    public GcsStorageOperator(GcsStorageProperties gcsStorageProperties) {
        super(gcsStorageProperties.getResourceUploadPath());
        this.bucketName = gcsStorageProperties.getBucketName();
        this.gcsStorage = (Storage)((StorageOptions.Builder)StorageOptions.newBuilder().setCredentials((Credentials)ServiceAccountCredentials.fromStream((InputStream)Files.newInputStream(Paths.get(gcsStorageProperties.getCredential(), new String[0]), new OpenOption[0])))).build().getService();
        this.checkBucketNameExists(this.bucketName);
    }

    public String getStorageBaseDirectory() {
        if (this.resourceBaseAbsolutePath.startsWith("/")) {
            log.warn("{} -> {} should not start with / in Gcs", (Object)"resource.storage.upload.base.path", (Object)this.resourceBaseAbsolutePath);
            return this.resourceBaseAbsolutePath.substring(1);
        }
        return this.getStorageBaseDirectory();
    }

    public void createStorageDir(String directoryAbsolutePath) {
        directoryAbsolutePath = this.transformAbsolutePathToGcsKey(directoryAbsolutePath);
        if (this.exists(directoryAbsolutePath)) {
            throw new FileAlreadyExistsException("directory: " + directoryAbsolutePath + " already exists");
        }
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)BlobId.of((String)this.bucketName, (String)directoryAbsolutePath)).build();
        this.gcsStorage.create(blobInfo, "".getBytes(StandardCharsets.UTF_8), new Storage.BlobTargetOption[0]);
    }

    public void download(String srcFilePath, String dstFilePath, boolean overwrite) {
        srcFilePath = this.transformAbsolutePathToGcsKey(srcFilePath);
        File dstFile = new File(dstFilePath);
        if (dstFile.isDirectory()) {
            Files.delete(dstFile.toPath());
        } else {
            FileUtils.createDirectoryWithPermission((Path)dstFile.getParentFile().toPath(), (Set)FileUtils.PERMISSION_755);
        }
        Blob blob = this.gcsStorage.get(BlobId.of((String)this.bucketName, (String)srcFilePath));
        blob.downloadTo(Paths.get(dstFilePath, new String[0]));
    }

    public boolean exists(String fullName) {
        Blob blob = this.gcsStorage.get(BlobId.of((String)this.bucketName, (String)(fullName = this.transformAbsolutePathToGcsKey(fullName))));
        return blob != null && blob.exists(new Blob.BlobSourceOption[0]);
    }

    public void delete(String filePath, boolean recursive) {
        filePath = this.transformAbsolutePathToGcsKey(filePath);
        if (this.exists(filePath)) {
            this.gcsStorage.delete(BlobId.of((String)this.bucketName, (String)filePath));
        }
    }

    public void copy(String srcPath, String dstPath, boolean deleteSource, boolean overwrite) {
        srcPath = this.transformGcsKeyToAbsolutePath(srcPath);
        dstPath = this.transformGcsKeyToAbsolutePath(dstPath);
        BlobId source = BlobId.of((String)this.bucketName, (String)srcPath);
        BlobId target = BlobId.of((String)this.bucketName, (String)dstPath);
        this.gcsStorage.copy(Storage.CopyRequest.newBuilder().setSource(source).setTarget(target).build());
        if (deleteSource) {
            this.gcsStorage.delete(source);
        }
    }

    public void upload(String srcFile, String dstPath, boolean deleteSource, boolean overwrite) {
        dstPath = this.transformAbsolutePathToGcsKey(dstPath);
        if (this.exists(dstPath) && !overwrite) {
            throw new FileAlreadyExistsException("file: " + dstPath + " already exists");
        }
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)BlobId.of((String)this.bucketName, (String)dstPath)).build();
        Path srcPath = Paths.get(srcFile, new String[0]);
        this.gcsStorage.create(blobInfo, Files.readAllBytes(srcPath), new Storage.BlobTargetOption[0]);
        if (deleteSource) {
            Files.delete(srcPath);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public List<String> fetchFileContent(String filePath, int skipLineNums, int limit) {
        filePath = this.transformAbsolutePathToGcsKey(filePath);
        if (StringUtils.isBlank((CharSequence)filePath)) {
            log.error("file path:{} is blank", (Object)filePath);
            return Collections.emptyList();
        }
        Blob blob = this.gcsStorage.get(BlobId.of((String)this.bucketName, (String)filePath));
        try (BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(blob.getContent(new Blob.BlobSourceOption[0]))));){
            Stream<String> stream = bufferedReader.lines().skip(skipLineNums).limit(limit);
            List<String> list = stream.collect(Collectors.toList());
            return list;
        }
    }

    @Override
    public void close() throws IOException {
        if (this.gcsStorage != null) {
            this.gcsStorage.close();
        }
    }

    public List<StorageEntity> listStorageEntity(String resourceAbsolutePath) {
        resourceAbsolutePath = this.transformAbsolutePathToGcsKey(resourceAbsolutePath);
        Page blobs = this.gcsStorage.list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)resourceAbsolutePath)});
        ArrayList<StorageEntity> storageEntities = new ArrayList<StorageEntity>();
        blobs.iterateAll().forEach(blob -> storageEntities.add(this.transformBlobToStorageEntity((Blob)blob)));
        return storageEntities;
    }

    public List<StorageEntity> listFileStorageEntityRecursively(String resourceAbsolutePath) {
        resourceAbsolutePath = this.transformAbsolutePathToGcsKey(resourceAbsolutePath);
        HashSet<String> visited = new HashSet<String>();
        ArrayList<StorageEntity> storageEntityList = new ArrayList<StorageEntity>();
        LinkedList<String> foldersToFetch = new LinkedList<String>();
        foldersToFetch.addLast(resourceAbsolutePath);
        while (!foldersToFetch.isEmpty()) {
            String pathToExplore = (String)foldersToFetch.pop();
            visited.add(pathToExplore);
            List<StorageEntity> tempList = this.listStorageEntity(pathToExplore);
            for (StorageEntity temp : tempList) {
                if (!temp.isDirectory() || visited.contains(temp.getFullName())) continue;
                foldersToFetch.add(temp.getFullName());
            }
            storageEntityList.addAll(tempList);
        }
        return storageEntityList;
    }

    public StorageEntity getStorageEntity(String resourceAbsolutePath) {
        resourceAbsolutePath = this.transformAbsolutePathToGcsKey(resourceAbsolutePath);
        Blob blob = this.gcsStorage.get(BlobId.of((String)this.bucketName, (String)resourceAbsolutePath));
        return this.transformBlobToStorageEntity(blob);
    }

    private void checkBucketNameExists(String bucketName) {
        if (StringUtils.isBlank((CharSequence)bucketName)) {
            throw new IllegalArgumentException("resource.google.cloud.storage.bucket.name is blank");
        }
        boolean exist = false;
        for (Bucket bucket : this.gcsStorage.list(new Storage.BucketListOption[0]).iterateAll()) {
            if (!bucketName.equals(bucket.getName())) continue;
            exist = true;
            break;
        }
        if (!exist) {
            throw new IllegalArgumentException("bucketName: " + bucketName + " is not exists, you need to create them by yourself");
        }
        log.info("bucketName: {} has been found", (Object)bucketName);
    }

    private StorageEntity transformBlobToStorageEntity(Blob blob) {
        String absolutePath = this.transformGcsKeyToAbsolutePath(blob.getName());
        ResourceMetadata resourceMetaData = this.getResourceMetaData(absolutePath);
        StorageEntity entity = new StorageEntity();
        entity.setFileName(new File(absolutePath).getName());
        entity.setFullName(absolutePath);
        entity.setDirectory(resourceMetaData.isDirectory());
        entity.setType(resourceMetaData.getResourceType());
        entity.setSize(blob.getSize().longValue());
        entity.setRelativePath(resourceMetaData.getResourceRelativePath());
        entity.setCreateTime(Date.from(blob.getCreateTimeOffsetDateTime().toInstant()));
        entity.setUpdateTime(Date.from(blob.getUpdateTimeOffsetDateTime().toInstant()));
        return entity;
    }

    private String transformAbsolutePathToGcsKey(String absolutePath) {
        ResourceMetadata resourceMetaData = this.getResourceMetaData(absolutePath);
        if (resourceMetaData.isDirectory()) {
            return FileUtils.concatFilePath((String[])new String[]{absolutePath, "/"});
        }
        return absolutePath;
    }

    private String transformGcsKeyToAbsolutePath(String gcsKey) {
        if (gcsKey.endsWith("/")) {
            return gcsKey.substring(0, gcsKey.length() - 1);
        }
        return gcsKey;
    }
}

