/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.text;

import java.util.Arrays;
import java.util.function.IntUnaryOperator;
import org.apache.commons.lang3.ArraySorter;
import org.apache.commons.text.CharacterPredicate;
import org.apache.commons.text.RandomStringGenerator;
import org.apache.commons.text.TextRandomProvider;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

class RandomStringGeneratorTest {
    private static final CharacterPredicate A_FILTER = codePoint -> codePoint == 97;
    private static final CharacterPredicate B_FILTER = codePoint -> codePoint == 98;

    RandomStringGeneratorTest() {
    }

    private static int codePointLength(String s) {
        return s.codePointCount(0, s.length());
    }

    @Test
    void testBadMaximumCodePoint() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> RandomStringGenerator.builder().withinRange(0, 0x110000));
    }

    @Test
    void testBadMinAndMax() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> RandomStringGenerator.builder().withinRange(2, 1));
    }

    @Test
    void testBadMinimumCodePoint() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> RandomStringGenerator.builder().withinRange(-1, 1));
    }

    @Test
    void testBuildDeprecated() {
        RandomStringGenerator.Builder builder = RandomStringGenerator.builder().withinRange(97, 122).filteredBy(new CharacterPredicate[]{A_FILTER});
        String str = builder.filteredBy(new CharacterPredicate[]{B_FILTER}).build().generate(100);
        for (char c : str.toCharArray()) {
            Assertions.assertEquals((char)'b', (char)c);
        }
    }

    @Test
    void testChangeOfFilter() {
        RandomStringGenerator.Builder builder = RandomStringGenerator.builder().withinRange(97, 122).filteredBy(new CharacterPredicate[]{A_FILTER});
        String str = builder.filteredBy(new CharacterPredicate[]{B_FILTER}).get().generate(100);
        for (char c : str.toCharArray()) {
            Assertions.assertEquals((char)'b', (char)c);
        }
    }

    @Test
    void testGenerateMinMaxLength() {
        boolean minLength = false;
        int maxLength = 3;
        RandomStringGenerator generator = RandomStringGenerator.builder().get();
        String str = generator.generate(0, 3);
        int codePointLength = RandomStringGeneratorTest.codePointLength(str);
        Assertions.assertTrue((codePointLength >= 0 && codePointLength <= 3 ? 1 : 0) != 0);
    }

    @Test
    void testGenerateMinMaxLengthInvalidLength() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> {
            RandomStringGenerator generator = RandomStringGenerator.builder().get();
            generator.generate(-1, 0);
        });
    }

    @Test
    void testGenerateMinMaxLengthMinGreaterThanMax() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> {
            RandomStringGenerator generator = RandomStringGenerator.builder().get();
            generator.generate(1, 0);
        });
    }

    @Test
    void testGenerateTakingIntThrowsNullPointerException() {
        Assertions.assertThrowsExactly(NullPointerException.class, () -> {
            RandomStringGenerator.Builder randomStringGeneratorBuilder = RandomStringGenerator.builder();
            CharacterPredicate[] characterPredicateArray = new CharacterPredicate[2];
            randomStringGeneratorBuilder.filteredBy(characterPredicateArray);
            RandomStringGenerator randomStringGenerator = randomStringGeneratorBuilder.get();
            randomStringGenerator.generate(18);
        });
    }

    @Test
    void testInvalidLength() {
        Assertions.assertThrowsExactly(IllegalArgumentException.class, () -> RandomStringGenerator.builder().get().generate(-1));
    }

    @Test
    void testMultipleFilters() {
        String str = RandomStringGenerator.builder().withinRange(97, 100).filteredBy(new CharacterPredicate[]{A_FILTER, B_FILTER}).get().generate(5000);
        boolean aFound = false;
        boolean bFound = false;
        for (char c : str.toCharArray()) {
            if (c == 'a') {
                aFound = true;
                continue;
            }
            if (c == 'b') {
                bFound = true;
                continue;
            }
            Assertions.fail((String)"Invalid character");
        }
        Assertions.assertTrue((aFound && bFound ? 1 : 0) != 0);
    }

    @Test
    void testNoLoneSurrogates() {
        int length = 5000;
        String str = RandomStringGenerator.builder().get().generate(5000);
        char lastChar = str.charAt(0);
        for (int i = 1; i < str.length(); ++i) {
            char c = str.charAt(i);
            if (Character.isLowSurrogate(c)) {
                Assertions.assertTrue((boolean)Character.isHighSurrogate(lastChar));
            }
            if (Character.isHighSurrogate(lastChar)) {
                Assertions.assertTrue((boolean)Character.isLowSurrogate(c));
            }
            if (Character.isHighSurrogate(c)) {
                Assertions.assertTrue((i + 1 < str.length() ? 1 : 0) != 0);
            }
            lastChar = c;
        }
    }

    @Test
    void testNoPrivateCharacters() {
        int codePoint;
        int startOfPrivateBMPChars = 57344;
        String str = RandomStringGenerator.builder().withinRange(57344, 65535).get().generate(5000);
        int i = 0;
        do {
            Assertions.assertFalse((Character.getType(codePoint = str.codePointAt(i)) == 18 ? 1 : 0) != 0);
        } while ((i += Character.charCount(codePoint)) < str.length());
    }

    @Test
    void testPasswordExample() {
        char[] punctuation = ArraySorter.sort((char[])new char[]{'!', '\"', '#', '$', '&', '\'', '(', ')', ',', '.', ':', ';', '?', '@', '[', '\\', ']', '^', '_', '`', '{', '|', '}', '~'});
        RandomStringGenerator generator = RandomStringGenerator.builder().setAccumulate(true).withinRange(97, 122).withinRange(65, 90).withinRange(48, 57).selectFrom(punctuation).get();
        String randomText = generator.generate(10);
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue((Character.isLetter(c) || Character.isDigit(c) || Arrays.binarySearch(punctuation, c) >= 0 ? 1 : 0) != 0);
        }
    }

    @Test
    void testRemoveFilters() {
        RandomStringGenerator.Builder builder = RandomStringGenerator.builder().withinRange(97, 122).filteredBy(new CharacterPredicate[]{A_FILTER});
        builder.filteredBy(new CharacterPredicate[0]);
        String str = builder.get().generate(100);
        for (char c : str.toCharArray()) {
            if (c == 'a') continue;
            return;
        }
        Assertions.fail((String)"Filter appears to have remained in place");
    }

    @Test
    void testSelectFromCharArray() {
        String str = "abc";
        char[] charArray = "abc".toCharArray();
        RandomStringGenerator generator = RandomStringGenerator.builder().selectFrom(charArray).get();
        String randomText = generator.generate(5);
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue(("abc".indexOf(c) != -1 ? 1 : 0) != 0);
        }
    }

    @Test
    void testSelectFromCharVarargs() {
        String str = "abc";
        RandomStringGenerator generator = RandomStringGenerator.builder().selectFrom(new char[]{'a', 'b', 'c'}).get();
        String randomText = generator.generate(5);
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue(("abc".indexOf(c) != -1 ? 1 : 0) != 0);
        }
    }

    @ParameterizedTest
    @ValueSource(booleans={false, true})
    void testSelectFromCharVarargs2(boolean accumulate) {
        String str = "abcde";
        RandomStringGenerator generator = RandomStringGenerator.builder().setAccumulate(accumulate).selectFrom(new char[0]).selectFrom(null).selectFrom(new char[]{'a', 'b'}).selectFrom(new char[]{'a', 'b', 'c'}).selectFrom(new char[]{'a', 'b', 'c', 'd'}).selectFrom(new char[]{'a', 'b', 'c', 'd', 'e'}).get();
        String randomText = generator.generate(10);
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue(("abcde".indexOf(c) != -1 ? 1 : 0) != 0);
        }
    }

    @ParameterizedTest
    @ValueSource(booleans={false, true})
    void testSelectFromCharVarargs3(boolean accumulate) {
        String str = "abcde";
        RandomStringGenerator generator = RandomStringGenerator.builder().setAccumulate(accumulate).selectFrom(new char[]{'a', 'b', 'c', 'd', 'e'}).selectFrom(new char[]{'a', 'b', 'c', 'd'}).selectFrom(new char[]{'a', 'b', 'c'}).selectFrom(new char[]{'a', 'b'}).selectFrom(null).selectFrom(new char[0]).get();
        String randomText = generator.generate(10);
        for (char c : randomText.toCharArray()) {
            Assertions.assertEquals((Object)accumulate, (Object)("abcde".indexOf(c) != -1 ? 1 : 0));
        }
    }

    @Test
    void testSelectFromCharVarargSize1() {
        RandomStringGenerator generator = RandomStringGenerator.builder().selectFrom(new char[]{'a'}).get();
        String randomText = generator.generate(5);
        for (char c : randomText.toCharArray()) {
            Assertions.assertEquals((char)'a', (char)c);
        }
    }

    @Test
    void testSelectFromEmptyCharVarargs() {
        RandomStringGenerator generator = RandomStringGenerator.builder().selectFrom(new char[0]).get();
        String randomText = generator.generate(5);
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue((c >= '\u0000' && c <= '\u10ffff' ? 1 : 0) != 0);
        }
    }

    @Test
    void testSelectFromNullCharVarargs() {
        int length = 5;
        RandomStringGenerator generator = RandomStringGenerator.builder().selectFrom(null).get();
        String randomText = generator.generate(5);
        Assertions.assertEquals((int)5, (int)RandomStringGeneratorTest.codePointLength(randomText));
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue((c >= '\u0000' && c <= '\u10ffff' ? 1 : 0) != 0);
        }
        RandomStringGenerator.Builder builder = RandomStringGenerator.builder().selectFrom(new char[]{'a'});
        generator = builder.get();
        randomText = generator.generate(5);
        for (char c : randomText.toCharArray()) {
            Assertions.assertEquals((char)'a', (char)c);
        }
        generator = builder.selectFrom(null).get();
        randomText = generator.generate(5);
        Assertions.assertEquals((int)5, (int)RandomStringGeneratorTest.codePointLength(randomText));
        for (char c : randomText.toCharArray()) {
            Assertions.assertTrue((c >= '\u0000' && c <= '\u10ffff' ? 1 : 0) != 0);
        }
    }

    @Test
    void testSetLength() {
        int length = 99;
        RandomStringGenerator generator = RandomStringGenerator.builder().get();
        String str = generator.generate(99);
        Assertions.assertEquals((int)99, (int)RandomStringGeneratorTest.codePointLength(str));
    }

    @Test
    void testUsingRandomIntUnaryOperator() {
        int testChar = 97;
        IntUnaryOperator testRandom = n -> 97;
        String str = RandomStringGenerator.builder().usingRandom(testRandom).get().generate(10);
        for (char c : str.toCharArray()) {
            Assertions.assertEquals((char)'a', (char)c);
        }
    }

    @Test
    void testUsingRandomTextRandomProvider() {
        int testChar = 97;
        TextRandomProvider testRandom = n -> 97;
        String str = RandomStringGenerator.builder().usingRandom(testRandom).get().generate(10);
        for (char c : str.toCharArray()) {
            Assertions.assertEquals((char)'a', (char)c);
        }
    }

    @Test
    void testWithinMultipleRanges() {
        int codePoint;
        int length = 5000;
        char[][] pairs = new char[][]{{'a', 'z'}, {'0', '9'}};
        RandomStringGenerator generator = RandomStringGenerator.builder().withinRange((char[][])new char[0][]).withinRange((char[][])null).withinRange((char[][])pairs).get();
        String str = generator.generate(5000);
        int minimumCodePoint = 0;
        int maximumCodePoint = 0;
        for (char[] pair : pairs) {
            minimumCodePoint = Math.min(minimumCodePoint, pair[0]);
            maximumCodePoint = Math.max(maximumCodePoint, pair[1]);
        }
        int i = 0;
        do {
            Assertions.assertTrue(((codePoint = str.codePointAt(i)) >= minimumCodePoint && codePoint <= maximumCodePoint ? 1 : 0) != 0);
        } while ((i += Character.charCount(codePoint)) < str.length());
    }

    @Test
    void testWithinRange() {
        int codePoint;
        int length = 5000;
        int minimumCodePoint = 97;
        int maximumCodePoint = 122;
        RandomStringGenerator generator = RandomStringGenerator.builder().withinRange(97, 122).get();
        String str = generator.generate(5000);
        int i = 0;
        do {
            Assertions.assertTrue(((codePoint = str.codePointAt(i)) >= 97 && codePoint <= 122 ? 1 : 0) != 0);
        } while ((i += Character.charCount(codePoint)) < str.length());
    }

    @Test
    void testZeroLength() {
        RandomStringGenerator generator = RandomStringGenerator.builder().get();
        Assertions.assertEquals((Object)"", (Object)generator.generate(0));
    }
}

