#!/usr/bin/env -S perl -CSDA

# Copyright 2021 Alin Mr. <almr.oss@outlook.com>. Licensed under the MIT license (https://opensource.org/licenses/MIT).

package Getopt::Auto::Long::Usage;
use v5.28.0;
use strict; use warnings;

=head1 NAME

C<Getopt::Auto::Long::Usage> - generate usage strings from Getopt::Long specs

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.01';

=head1 SYNOPSIS

This is a pure perl module that generates simple usage / help messages by parsing L<Getopt::Long> argument specs (and optionally using provided descriptions).

      print getoptlong2usage( Getopt_Long => \@conf [, ...] )

=head1 DESCRIPTION

C<Getopt::Auto::Long::Usage> strives to be compatible with L<Getopt::LongUsage>. In particular, it does not require supplementing existing arglist specs with additional data (e.g. descriptions are optional). However, the goal is to provide maximum maintainability with the least amount of code, not to achieve complete L<Getopt::Long> coverage. So, there are some differences:

=over 4

=item * the generated usage clearly distinguishes boolean flags from arguments requiring an option, and prints type information for the latter. For negatable boolean options (C<longopt|s!>), it will print the corresponding C<--no-longopt> flag (but not C<--no-s>).

=item * there are no dependencies; the main function can be copied directly into your source code, if necessary

=item * it does not attempt to parse C<GetOptions()> abbreviated / case-insensitive options, and in fact recommends that you disable those when using C<Getopt::Long> for maintainability and predictability. One shortopt + one (or several) longopts, explicitly specified, will avoid nasty surprises (plus, suppose you decide to rewrite the code in some other language...)

=back

The following example should print the generated help message either to stdout, if requested (C<--help>) or to stderr, if argument parsing fails.

    use Getopt::Auto::Long::Usage;
    use Getopt::Long;
    my @getoptargs = qw{ help
                         delim:s
                         eval|e!
                       };
    my %_O_; my @getoptconf = (\%_O_, @getoptargs);

    sub usage {
      my ($rc) = @_;
      my @dsc = ( delim => 'instead of newline' );
      print getoptlong2usage(
        Getopt_Long => \@getoptconf, # all others optional
        cli_use => "Arguments: [OPTION]...\nOptions:";
        footer => "No other arguments may be supplied"
        descriptions => \@dsc
      );
      exit $rc if defined( $rc );
    }

    Getopt::Long::Configure( qw(
      no_ignore_case no_auto_abbrev no_getopt_compat
      gnu_compat bundling
    ));
    unless( GetOptions( @getoptconf ) ) {
      local *STDOUT = *STDERR; usage 1;
    }
    usage 0 if( $_O_{ help } );

=head1 EXPORT

=over 4

=item * C<getoptlong2usage>

=back

=cut

use Exporter qw(import);
our @EXPORT = qw(getoptlong2usage);

=head1 FUNCTIONS

=head2 getoptlong2usage

  $usage = getoptlong2usage( Getopt_Long => \@getoptconf [,
    descriptions => \@dsc,
    cli_use => $cli_use,
    indent => $n,
    pfx => $prefix ] )

C<@getoptconf> is an arrayref containing all the arguments you would supply to C<GetOptions()>, including the initial hashref in which C<GetOptions()> stores results (and which is ignored). It's easiest to define C<@getoptconf> separately and reuse it for both calls. See the synopsis for an example.

=cut

sub getoptlong2usage {
  my %_O_ = @_; $_O_{ descriptions } //= [];
  $_O_{ $_ } //= '' for qw{ pfx cli_use footer };
  my ($conf, $pfx ) = @_O_{ qw{ Getopt_Long pfx }};
  my %dsc = @{ $_O_{ descriptions } };

  my $opt2dash = sub { length $_ == 1 ? "-$_" : "--$_"; };
  my $finalnl = sub { my ($s) = @_; $s .= "\n" unless $s =~ /(^|\n)$/; $s; };
  # my $uniq = sub { my %cn; return grep { ! $cn{ $_ }++ } @_; };  # for auto_abbrev

  my $out = &$finalnl( $_O_{ cli_use } );
  $pfx = (' ' x $_O_{ indent }) if defined( $_O_{ indent } );

  my %t2p = (s => 'STR', i => 'INT', f => 'REAL', o => '[0[x|b]]INT', '+' => 'repeated...');  # TODO: :number, = t [desttype] [repeat]

  for( @{ $conf }[ 1..$#$conf ] ) {
    my $a = $_; my $isneg = 0; my $t = ''; my $isopt = 0;
    if    (/(.*)!$/)    { $a = $1; $isneg = 1; }
    if    (/(.*)\+$/)   { $a = $1; $t = '+'; $isopt = 1; }
    elsif (/(.*):(.*)/) { $a = $1; $t = $2; $isopt = 1; }
    elsif (/(.*)=(.*)/) { $a = $1; $t = $2; }

    $a =~ qr{ (^|\|) ( (?<long> [^|]{2,}) ($|\|) ) }x; my $along = $+{long};
    my @aa = split( /\|/, $a );
    # @aa = &$uniq( @aa, substr( $along, 0, 1 ) ) if length $along;  # handle auto_abbrev
    $out .= $pfx . join( ' | ', map( &$opt2dash, @aa ) );
    if( length $t ) {
      $t = $t2p{ $t } // "ARG:$t"; $t = "[$t]" if $isopt;
      $out .= " $t";
    } else {
      if( $isneg ) { $out .= "\n$pfx--no-" . $along if length $along; }
    }
    $out .= ": $dsc{ $along }" if length $along and $dsc{ $along };
    $out .= "\n";
  }
  return $out . &$finalnl( $_O_{ footer } );
}

1;

=head1 SOURCE

The source code repository can be found at L<https://gitlab.com/kstr0k/perl-getopt-auto-long-usage.git>.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Getopt::Auto::Long::Usage

=head1 AUTHOR

Alin Mr., C<< <almr.oss at outlook.com> >>

=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2021 by Alin Mr.

This is free software, licensed under:

  The MIT (X11) License

=cut

# vi: set ft=perl:
