package App::Manoc::Controller::Server;
#ABSTRACT: Server controller

use Moose;

our $VERSION = '2.99.4'; ##TRIAL VERSION

use namespace::autoclean;

BEGIN { extends 'Catalyst::Controller'; }

with 'App::Manoc::ControllerRole::CommonCRUD';

use App::Manoc::Form::Server;
use App::Manoc::Form::Server::Decommission;
use App::Manoc::Form::ServerNWInfo;

__PACKAGE__->config(
    # define PathPart
    action => {
        setup => {
            PathPart => 'server',
        }
    },
    class            => 'ManocDB::Server',
    form_class       => 'App::Manoc::Form::Server',
    view_object_perm => undef,
    json_columns     => [ 'id', 'name' ],

    find_object_options => { prefetch => { installed_sw_pkgs => 'software_pkg' } },

    create_page_title => 'Create server',
    edit_page_title   => 'Edit server',
);


before 'create' => sub {
    my ( $self, $c ) = @_;

    my $serverhw_id = $c->req->query_parameters->{'serverhw'};
    my $vm_id       = $c->req->query_parameters->{'vm'};
    my %form_defaults;

    if ( defined($serverhw_id) ) {
        $c->log->debug("new server using hardware $serverhw_id") if $c->debug;
        $form_defaults{serverhw} = $serverhw_id;
        $form_defaults{type}     = 'p';
    }
    if ( defined($vm_id) ) {
        $c->log->debug("new server using vm $vm_id") if $c->debug;
        $form_defaults{vm}   = $vm_id;
        $form_defaults{type} = 'v';
    }
    %form_defaults and
        $c->stash( form_defaults => \%form_defaults );
};


before 'edit' => sub {
    my ( $self, $c ) = @_;

    my $object    = $c->stash->{object};
    my $object_pk = $c->stash->{object_pk};

    # decommissioned objects cannot be edited
    if ( $object->decommissioned ) {
        $c->res->redirect( $c->uri_for_action( 'server/view', [$object_pk] ) );
        $c->detach();
    }
};


sub decommission : Chained('object') : PathPart('decommission') : Args(0) {
    my ( $self, $c ) = @_;

    $c->require_permission( $c->stash->{object}, 'edit' );

    my $form = App::Manoc::Form::Server::Decommission->new( { ctx => $c } );

    $c->stash(
        form   => $form,
        action => $c->uri_for( $c->action, $c->req->captures ),
    );
    return unless $form->process(
        item   => $c->stash->{object},
        params => $c->req->parameters,
    );

    $c->response->redirect( $c->uri_for_action( 'server/view', [ $c->stash->{object_pk} ] ) );
    $c->detach();
}


sub restore : Chained('object') : PathPart('restore') : Args(0) {
    my ( $self, $c ) = @_;

    my $server = $c->stash->{object};
    $c->require_permission( $server, 'edit' );

    if ( !$server->decommissioned ) {
        $c->response->redirect( $c->uri_for_action( 'server/view', [ $server->id ] ) );
        $c->detach();
    }

    if ( $c->req->method eq 'POST' ) {
        $server->restore;
        $server->update();
        $c->flash( message => "Server restored" );
        $c->response->redirect( $c->uri_for_action( 'server/view', [ $server->id ] ) );
        $c->detach();
    }

    # show confirm page
    $c->stash(
        title           => 'Restore server',
        confirm_message => 'Restore decommissioned server ' . $server->label . '?',
        template        => 'generic_confirm.tt',
    );
}


sub nwinfo : Chained('object') : PathPart('nwinfo') : Args(0) {
    my ( $self, $c ) = @_;

    $c->require_permission( $c->stash->{object}, 'netwalker_config' );

    my $server_id = $c->stash->{object_pk};

    my $nwinfo = $c->model('ManocDB::ServerNWinfo')->find($server_id);
    $nwinfo or $nwinfo = $c->model('ManocDB::ServerNWInfo')->new_result( {} );

    my $form = App::Manoc::Form::ServerNWInfo->new(
        {
            server => $server_id,
            ctx    => $c,
        }
    );
    $c->stash( form => $form );
    return unless $form->process(
        params => $c->req->params,
        item   => $nwinfo
    );

    $c->response->redirect( $c->uri_for_action( 'server/view', [$server_id] ) );
    $c->detach();
}


sub refresh : Chained('object') : PathPart('refresh') : Args(0) {
    my ( $self, $c ) = @_;
    my $id = $c->stash->{object}->id;

    my $config = App::Manoc::Netwalker::Config->new( $c->config->{Netwalker} || {} );
    my $client = App::Manoc::Netwalker::ControlClient->new( config => $config );

    my $status = $client->enqueue_server($id);

    if ( !$status ) {
        $c->flash( error_msg => "An error occurred while scheduling server refresh" );
    }
    else {
        $c->flash( message => "Device refresh scheduled" );
    }

    $c->response->redirect( $c->uri_for_action( '/server/view', [$id] ) );
    $c->detach();
}


sub update_from_nwinfo : Chained('object') : PathPart('from_nwinfo') : Args(0) {
    my ( $self, $c ) = @_;

    my $server = $c->stash->{object};
    $c->require_permission( $server, 'edit' );

    my $response = {};
    $response->{success} = 0;

    if ( !$server->decommissioned &&
        defined( $server->netwalker_info ) &&
        $c->req->method eq 'POST' )
    {
        my $nwinfo = $server->netwalker_info;
        my $what   = lc( $c->req->params->{what} );
        $what eq 'hostname' and $server->hostname( $nwinfo->name );
        $what eq 'os'       and $server->os( $nwinfo->os );
        $what eq 'os_ver'   and $server->os_ver( $nwinfo->os_ver );
        $server->update();
        $response->{success} = 1;
    }

    $c->stash( json_data => $response );
    $c->forward('View::JSON');
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::Server - Server controller

=head1 VERSION

version 2.99.4

=head1 ACTIONS

=head2 create

=head2 edit

=head2 decommission

=head2 restore

=head2 nwinfo

=head2 refresh

=head2 update_from_nwinfo

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
